/**
* \file: errmemd.c
*
* Implementation of the main function of the error memory daemon.
* 
* This file implements the main function of the error memory daemon.
* This includes the parsing of the command line options, the handling
* of the backends and the implementation of the server interface.
*
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#include <arpa/inet.h>
#include <netinet/in.h>
#include <regex.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <ctype.h>
#include <sys/un.h>
#include <sys/epoll.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <syslog.h>
#include <systemd/sd-daemon.h>

#include "errmemd.h"
#include "errmem_debug.h"
#include "errmem_backend.h"
#include "errmem_lib.h"
#include "errmem_interface.h"
#include "errmem_erase.h"

#define LONGOPT_BASE  0x500 /* to identify input of long options */
#define LONGOPT_END   (LONGOPT_BASE + 31) /* current end of long options */
#define EXPECTED      0xA /* necessary parameter to create epoll instance */
//----------------------------------------------------------------------
// command line interface

// USAGE - help text for the command line interface
#define USAGE_DLT                                                             \
  "    -g --dlt-output    Interface to DLT will be created.\n"                \
  "                       If this option is set the daemon, continuously\n"   \
  "                       writes the current content of the error memory\n"   \
  "                       frontend to DLT interface, in case it has no\n"     \
  "                       persistent storages available.\n"                   \
  "                       As long as at least one configured persistent\n"    \
  "                       storage is available, this option has no effect.\n" \
  "                       \n"
#ifdef _ARCH_x86_
#define USAGE_ARCH                                                            \
  "\nx86 specific options:\n"                                                 \
  "    -x           debug without daemonizing\n"                              \
  "    -y [<path>]  inject messages into the error memory\n"
#else // ifdef _ARCH_x86_
#define USAGE_ARCH
#endif // ifdef _ARCH_x86_

#define USAGE                                                                 \
  "usage: Run the error memory backend as logging and serving daemon:\n\n"    \
  " errmemd -d [<device> [-l <low> -u [<high>]]] [[-s | -r | -f] <storage>"   \
  " -b <blocksize> -c <blockcount> -e <persistent blocks>"                    \
  " -o <access options> -w <write chunk size> -E <erase wait time>"           \
  " -R <retries>]... -i\n\n"                                                  \
                                                                              \
  "usage: Perform an implemented smoketest:\n\n"                              \
  " errmemd -m [<device>]\n\n"                                                \
                                                                              \
  "usage: Use the error memory backend as an application to access given\n"   \
  "       persistent storages for dumping or invalidating their content\n\n"  \
  " errmemd [-s|-r|-f ] <storage>  -C dump | ddump | erase [-n <number>]]..." \
  " -i\n\n"                                                                   \
                                                                              \
  "    -d --daemon [<device> [-l <low> [-u <high>]]]\n\n"                     \
  "       Run backend as daemon having <device> as the frontend.\n"           \
  "       The default device is /dev/errmem. Only in case another\n"          \
  "       device shall be used it has to be provided here.\n"                 \
  "       In case the -d option is ommitted errmemd can be run as an\n"       \
  "       application which executes a command provided by -C or --cmd on\n"  \
  "       the specified persitent storage (refer to -C or --cmd option).\n\n" \
  "    -l --watermark-low\n\n"                                                \
  "       Watermark low determines the amount of messages remaining\n"        \
  "       in the driver device. The driver starts to inform the\n"            \
  "       the system about available messages if this mark exceeds.\n\n"      \
  "    -u --watermark-high\n\n"                                               \
  "       Watermark high defines the amount of messages which instructs\n"    \
  "       the driver to always inform the system about available messages\n\n"\
  "    -s --block-storage <storage> -b <blocksize> -c <blockcount>"           \
  " -e <persistent blocks> -o[0|1|2|3] -w <write chunk size>\n"               \
  "    or\n"                                                                  \
  "    -r --flash-storage <storage> -b <blocksize> -c <blockcount>"           \
  " -e <persistent blocks> -o -w <write chunk size> -E <erase-wait-time>"     \
  " -R <retries>\n"                                                           \
  "    or\n"                                                                  \
  "    -f --file-storage <storage> -b <filesize> -c <number of files>"        \
  " -e <persistent files>\n"                                                  \
  "       Either block oriented persistent storage or log files to store\n"   \
  "       messages.\n"                                                        \
  "       Block oriented does not mean that the underlying physical device\n" \
  "       has to be handled as a block device rather than that the daemon\n"  \
  "       handles the persistent storage in number of <blockcount> blocks\n"  \
  "       each of size <blocksize> (see options -b --bs and -c --cnt).\n"     \
  "       The daemon does not write blockwise.\n"                             \
  "       As soon as data is available the daemon writes this data to the\n"  \
  "       device concerning a potentially provided write size. If a unit\n"   \
  "       of size blocksize is used, the daemon continues writing data to\n"  \
  "       the next unit of size blocksize and so on. A block is handled as\n" \
  "       entity with own meta data like block number, validity magic etc.\n" \
  "       Please note that the specification of <blocksize>, <blockcount>\n"  \
  "       and number of <persistent blocks> is mandatory when specifying a\n" \
  "       block oriented persistent storage. These definitions have to\n"     \
  "       follow directly after the device specification in order to define\n"\
  "       the assignment of those attributes to the persistent storages\n"    \
  "       (see the whole expression in brackets above).\n"                    \
  "       This is the same for file storages. File storages consist\n"        \
  "       of a configuration file and the corresponding storage files.\n"     \
  "       A file storage named log.txt will consist of a configuration file\n"\
  "       log.txt.cfg and the storage files log.txt_0 to log.txt_<n>.\n"      \
  "       In additon the daemon needs READ/WRITE/EXECUTE rights for the\n"    \
  "       specified persistent storages and the directories they are in\n"    \
  "       rescpectively, when these storages are accessed as files.\n"        \
  "       The daemon needs READ/WRITE rights for persistent storages\n"       \
  "       which are accessed as raw devices don't having a file system\n"     \
  "       mounted on them.\n\n"                                               \
  "       An arbitrary amount of persistent storages can be configured\n"     \
  "       using the options\n"                                                \
  "       -s or --block-storage or\n"                                         \
  "       -r or --flash-storage or\n"                                         \
  "       -f or --file-storage as often as desired.\n"                        \
  "       The option -r or --flash-storage has to be used to specify\n"       \
  "       a persistent storage which shall reside on a system which has\n"    \
  "       bare flash and do not use FTL (Flash Translation Layer).\n"         \
  "       These are e.g. raw NOR or raw NAND flash devices. This option\n"    \
  "       instructs the daemon to assume a device which has eraseblocks,\n"   \
  "       which may be written to, which may be read from or which may be\n"  \
  "       erased. In case such a persistent storage has to be accessed\n"     \
  "       without a filesystem (without option -o --file-access) the daemon\n"\
  "       takes care of the erasesize of eraseblocks and the write size\n"    \
  "       specified for this device as well as of the special handling\n"     \
  "       which is necessary for such devices.\n\n"                           \
  "       Please note that the use of bare NAND flashes without appropriate\n"\
  "       file system support (e.g. UBIFS or JFFS2) is not supported by the\n"\
  "       daemon. This is because the daemon doesn't implement this complex\n"\
  "       functionality do deal with issues like bad block handling or\n"     \
  "       wear-leveling which is essential for the usage of such devices.\n\n"\
  "       When using a raw NOR flash device to hold the persistent storage,\n"\
  "       the specified blocksize shall be the erasesize of the device or\n"  \
  "       at least a multiple of the erasesize and the write size shall\n"    \
  "       match exactly the write size of the flash device. For more\n"       \
  "       details see option -w --write-size.\n"                              \
  "       If file system support is specified for a bare NOR flash device\n"  \
  "       or a bare NAND flash device, the daemon relies on the abilities\n"  \
  "       of the file system to handle these constraints and issues coming\n" \
  "       along wiht the use of such bare flash devices.\n"                   \
  "       Additional access options provided by file system option -o or\n"   \
  "       --file-access are not supported for bare flash devices yet. For\n"  \
  "       more details refer to option -o --file-access\n\n"                  \
  "       The option -s or --block-storage shall be used for all other\n"     \
  "       devices which can hold a persistent storage and behave like a\n"    \
  "       block device. These are e.g. FTL devices like e.g. USB-Sticks,\n"   \
  "       eMMC,SD, SSD or other FTL devices have a built-in controller\n"     \
  "       which runs FTL firmware, hard disks, etc. This enables the\n"       \
  "       daemon to handle such devices as block devices with no special\n"   \
  "       constraints or device specific issues. The daemon manages these\n"  \
  "       devices block oriented as described at the beginning of this\n"     \
  "       paragraph.\n\n"                                                     \
  "       For both classes of devices, bare flash devices and block devices\n"\
  "       it is mandatory to provide the blocksize, the blockcount and the\n" \
  "       number of persistent blocks for each specified device. In case\n"   \
  "       file system support is active (option -o --file-access is set)\n"   \
  "       it is not necessary to provide a write size using the option -w \n" \
  "       or --write-size. If file system support is not active, it is\n"     \
  "       mandatory to provide the write-size.\n"                             \
  "       For bare flash devices the persistent storage has to be available\n"\
  "       either as partition (raw NOR flash access) or as file in a file\n"  \
  "       system when file system support is active.\n"                       \
  "       For block devices the persistent storage has to be available as a\n"\
  "       partition for raw block device access. In case of activated\n"      \
  "       file system support it depends on the specified access options.\n"  \
  "       In case creation / recreation is allowed the daemon will create\n"  \
  "       a not existing file otherwise it will ignore that specification.\n" \
  "       The following device definitions are supported:(short options)\n"   \
  "       x: mandatory; (x) optional ; (-) not set\n\n"                       \
  "  | s | r | b | c | e || o | none | 1 | 2 | 3 || w | E | R |Class|Acess|\n"\
  "  |=====================================================================\n"\
  "  |=====================================================================\n"\
  "  | x |(-)| x | x | x || x | (x)  |(x)|(x)|(x)||(x)|(-)|(-)|Block| File|\n"\
  "  |== |===|===|===|===||===|======|===|===|===||===|===|===|=====|======\n"\
  "  | x |(-)| x | x | x ||(-)| (-)  |(-)|(-)|(-)|| x |(-)|(-)|Block| Raw |\n"\
  "  |== |===|===|===|===||===|======|===|===|===||===|===|===|=====|======\n"\
  "  |(-)| x | x | x | x || x |  x   |(-)|(-)|(-)||(x)|(-)|(-)|Flash| File|\n"\
  "  |===|===|===|===|===||===|======|===|===|===||===|===|===|=====|=====|\n"\
  "  |(-)| x | x | x | x ||(-)| (-)  |(-)|(-)|(-)|| x |(x)|(x)| NOR | Raw |\n"\
  "  |===|===|===|===|===||===|======|===|===|===||===|===|===|=====|=====|\n"\
  "  |(-)| x | x | x | x || x |  x   |(-)|(-)|(-)|| x |(-)|(-)| NAND| Raw |\n"\
  "  |===|===|===|===|===||===|======|===|===|===||===|===|===|=====|=====|\n"\
  "       \n"                                                                 \
  "  | f |   | b | c | e ||   |      |   |   |   ||   |   |   |Class|Acess|\n"\
  "  |=====================================================================\n"\
  "  |=====================================================================\n"\
  "  | x |   | x | x | x ||   |      |   |   |   ||   |   |   |File | File|\n"\
  "  |== |===|===|===|===||===|======|===|===|===||===|===|===|=====|======\n"\
  "       \n"                                                                 \
  "    -b --bs <blocksize>\n\n"                                               \
  "       Specifies the size in bytes the daemon shall use as \n"             \
  "       the size for one block. The provided blocksize shall follow\n"      \
  "       the following rule: blocksize = 512 * 2 **(n-1).\n"                 \
  "       Other blocksizes will be rejected. In case of a bare flash device\n"\
  "       (option -r without option -o) the blocksize has to be the\n"        \
  "       erasesize of the underlying flash device or a multiple of it.\n"    \
  "       The daemon organizes the specified device in units (blocks) of\n"   \
  "       this size. This size can differ from the real blocksize of the\n"   \
  "       underlying physical block device.\n"                                \
  "       In case of a file storage the file size is specified. The real\n"   \
  "       filesize will be truncated to the last multiple of the blocksize\n" \
  "       used by the underlying file system which does not exceed the\n"     \
  "       specified filesize\n\n"                                             \
  "    -c --cnt <blockcount>\n\n"                                             \
  "       Specifies the number of blocks the daemon shall use.\n"             \
  "       Note that the device specified shall have at least the size of \n"  \
  "       <blockcount> * <blocks> bytes. Blocks will be overwritten if\n"     \
  "       necessary (due to a wrap around). This is when all available\n"     \
  "       blocks are in use and new messages have to be stored.\n"            \
  "       Blocks which are excepted from the overwriting mechanism are the\n" \
  "       so called non overwritable blocks (see -e option).\n\n"             \
  "       In case of a file storage the maximum number of files to create\n"  \
  "       is specified. Files will be indexed. The index is seperated by\n"   \
  "       a '_' (underscore) character from the filename.\n"                  \
  "       Example: log.txt ==> log.txt_0 (first file)\n"                      \
  "       The name of the configuration file is created by extending the\n"   \
  "       specified file storage name with \".cfg\"\n"                        \
  "       Example: log.txt ==> log.txt.cfg\n"                                 \
  "       If the maximum number of files is reached the oldest file will be\n"\
  "       deleted. An amount of persistent files will be kept as long as\n"   \
  "       the persistent file storage exists. These persistent files are\n"   \
  "       not touched when the oldest file has to be deleted.\n\n"            \
  "    -e --persistent <persistent blocks>\n\n"                               \
  "       Defines a number of blocks at the beginning of the block device\n"  \
  "       which will not be overwritten when a wrap around has to be done\n"  \
  "       to store new messages persistently because all available blocks\n"  \
  "       (see blockcount) are already in use. In case the content of a\n"    \
  "       storage is erased, the content of these as persistent defined\n"    \
  "       blocks is erased too. These blocks will then be reused.\n"          \
  "       Persistent blocks will be written only once and then keep their\n"  \
  "       content until the content of the storage is erased by command.\n"   \
  "       All other blocks will be overwritten as soon as the memory is\n"    \
  "       needed for new messages.\n\n"                                       \
  "       In case of a file storage this option specifies the amount of\n"    \
  "       files which will not be deleted during life time of the file\n"     \
  "       storage.\n\n"                                                       \
  "    -o --file-access <access options>\n\n"                                 \
  "       This option tells the daemon that the specified persistent block\n" \
  "       device resides in a file system. The daemon does not necessarily\n" \
  "       take care of the characteristics of the resource or physical\n"     \
  "       device specified. In other words the daemon assumes the existence\n"\
  "       of a file system which manages the storage access.\n"               \
  "       The daemon organizes the persistent storage blockwise. That means\n"\
  "       the daemon reads the storage in chunks of <blocksize> bytes\n"      \
  "       and messages will be written into a block until the block is used\n"\
  "       completely and no complete new message can be stored into it.\n"    \
  "       In consequence the next block will be used to store new data.\n"    \
  "       If all blocks are used the daemon does a wrap around and starts\n"  \
  "       to overwrite already used blocks beginning with the first block\n"  \
  "       which is not part of the area of persistent blocks (see option\n"   \
  "       -e --persistent blocks for more details).\n\n"                      \
  "       !! Don't mix up the physical block size of a physical storage\n"    \
  "       !! and the blocksize specified in the command line for a\n"         \
  "       !! persistent storage. They can differ and are not necessarily\n"   \
  "       !! dependent from each other. One convention is that the\n"         \
  "       !! blocksize shall follow the rule: blocksize = 512*2**(n-1).\n"    \
  "       !! Another important dependency is give when using raw NOR flash\n" \
  "       !! devices. In this case the blocksize shall be the erasesize\n"    \
  "       !! of the underlying device or at least a multiple of it to\n"      \
  "       !! satisfy the physical device characteristics.\n\n"                \
  "       This option allows providing additional optional access options.\n" \
  "       These access options are only supported for persistent storages\n"  \
  "       which are specified with option -s --block-storage.\n"              \
  "       Persistent storages specified with option -r --flash-storage have\n"\
  "       no support of access options.\n"                                    \
  "       (refer to options -s --block-storage and -r --flash-storage\n"      \
  "       for more details)\n"                                                \
  "       During startup the daemon validates the existence of storages,\n"   \
  "       the integrity of an existing storage relating to size information\n"\
  "       and the content of an existing storage. The results of these\n"     \
  "       checks determine how to handle a specified persistent storage.\n"   \
  "       Additonal access options instruct the daemon what it is allowed\n"  \
  "       to do. The daemon can be instructed to create not existing\n"       \
  "       persistent storages or to recreate persistent storages in case\n"   \
  "       validity checks bring up inconsistency between specification and\n" \
  "       the current layout of an existing persistent storage.\n"            \
  "       The daemon can be instructed to do a backup of an existing\n"       \
  "       persistent storage if one of the checks fails.\n"                   \
  "       Supported access options are:\n\n"                                  \
  "       parameter | access option ==> creation | recreation | backup\n"     \
  "       ============================================================\n"     \
  "       -o        |      none     ==>    -     |     -      |    -  \n"     \
  "       -o1       |       1       ==>    x     |     x      |    -  \n"     \
  "       -o2       |       2       ==>    -     |     -      |    x  \n"     \
  "       -o3       |       3       ==>    x     |     x      |    x  \n\n"   \
  "       -o   : No access option given ==> none\n"                           \
  "       In case a specified persistent storage does not exist, the daemon\n"\
  "       ignores this persistent storage. It will not be created.\n"         \
  "       In case a specified persistent storage already exists but the\n"    \
  "       size information (blocksize, block count) provided with the\n"      \
  "       command line differs from the size information read from the\n"     \
  "       persistent storage itself, the daemon will not recreate this\n"     \
  "       persistent storage. In case the existing persistent storage is\n"   \
  "       great enough to satisfy the specified size, the daemon will use\n"  \
  "       it, concerning these new parameters.\n"                             \
  "       In case the existing persistent storage is not great enough the\n"  \
  "       daemon will ignore this specified storage because of missing\n"     \
  "       creation rights.\n"                                                 \
  "       In case the size information of an existing persistent storage\n"   \
  "       provided by the file system differs from the size information\n"    \
  "       provided in the command line the behavior is identical to the\n"    \
  "       ones previously described.\n\n"                                     \
  "       -o1  : creation and recreation active\n"                            \
  "       Whenever the daemon comes to the conclusion that a specified\n"     \
  "       persistent storage has to be created or recreated due to failing\n" \
  "       checks, it tries to create the persistent storage. An already\n"    \
  "       existing persitent storage will be overwritten (recreation)\n\n"    \
  "       -o2  : backup active\n"                                             \
  "       Whenever the daemon comes to the conclusion that an existing\n"     \
  "       persistent storage is invalid for some reason, the daemon tries\n"  \
  "       to create a backup of this persistent storage. This access option\n"\
  "       does not allow to create or recreate a persistent storage in case\n"\
  "       of need. That means, whenever the daemon cannot reuse the\n"        \
  "       existing persistent storage, the storage will be backed up but\n"   \
  "       ignored for further use.\n\n"                                       \
  "       -o3  : creation, recreation and backup active.\n"                   \
  "       This access option combines access option 1 and 2.\n"               \
  "       Whenever the daemon comes to the result that a specified\n"         \
  "       persistent storage has to be backed up or created or recreated,\n"  \
  "       the daemon tries to execute the corresponding commands.\n\n"        \
  "       If the -o option is omitted the daemon assumes that the\n"          \
  "       persistent storage is a raw block device without any file\n"        \
  "       system mounted on it. It acesses the device with block access.\n"   \
  "       The daemon doesn't try to create or backup such persistent\n"       \
  "       storages. In case the specified persistent storage is not\n"        \
  "       available, it is ignored. If it is corrupted it is setup newly\n"   \
  "       for further use if possible.\n"                                     \
  "       For raw block devices (-o option omitted) specifying option\n"      \
  "       -w --write size is mandatory.\n\n"                                  \
  "    -w --write-size <write chunck size>\n\n"                               \
  "       Specifies the alignement in bytes with which data shall be\n"       \
  "       written to the device. Write requests will be arranged in that\n"   \
  "       way that each write request results in a write to a start address\n"\
  "       having the given alignment and having a length which is always a\n" \
  "       multiple of the given alignemnt.\n"                                 \
  "       This is to meet the characteristics of the underlying physical\n"   \
  "       device in order to pay attention for internal ECC computation.\n"   \
  "       This value should represent the write size to use for the \n"       \
  "       underlying physical device.\n"                                      \
  "       For raw block devices (-o option omitted) the specification of\n"   \
  "       this option is mandatory.\n\n"                                      \
  "    -E --erase-wait-time <time>\n\n"                                       \
  "       This option is meant to be used only for bare NOR flash device.\n"  \
  "       Time in seconds the daemon waits between two erase commands on\n"   \
  "       the same physical eraseblock of the bare NOR flash device.\n"       \
  "       If this option is ommitted the daemon uses a default value of 0s.\n"\
  "       The daemon uses CLOCKMONOTONIC_RAW clock id to get a timestamp.\n"  \
  "       Every physical eraseblock of the bare NOR flash device has its\n"   \
  "       own last erase time based on this timestamp. After power up or\n"   \
  "       reset this timestamp is 0. If it becomes necessary to erase\n"      \
  "       a complete eraseblock, the daemon waits until <time> is elapsed\n"  \
  "       before issuing the erase command to the device.\n"                  \
  "       Note: If you issue the erase command to a bare flash device\n"      \
  "             using the backend without -d option (client mode),\n"         \
  "             the daemon will not recognize this process and therefore\n"   \
  "             not either the time at which the physical block was erased.\n"\
  "             In such cases you have to guarantee the wait time on your\n"  \
  "             own. This includes the time between to erase commands sent\n" \
  "             to the bare flash device using -C option and the time\n"      \
  "             between an erase command using -C option and the daemon\n"    \
  "             startup.\n"                                                   \
  "             Furthermore this time incluences if necessary the storing\n"  \
  "             of messages from the error memory frontend into persistent\n" \
  "             storages. The daemon stores a message, retrieved from the\n"  \
  "             error memory frontend, consecutively into all specified\n"    \
  "             persistent storages. In case the scenario occurs that the\n"  \
  "             deamon needs to wait for <time> seconds or at least for a\n"  \
  "             part of this time intervall the storage process is stopped\n" \
  "             as long as this wait takes place.\n\n"                        \
  "    -R --erase-retry <retries>\n\n"                                        \
  "       This option is meant to be used only for bare NOR flash device.\n"  \
  "       This is the number of retries that the daemon uses to repeat\n"     \
  "       trying to erase an eraseblock in case any error occured.\n"         \
  "       Note: The wait time for storing messages can be in the worst-case\n"\
  "             retries * erase-wait-time.\n"                                 \
  "       In case this option is ommitted the default value is 0.\n\n"        \
  "    -C --cmd\n\n"                                                          \
  "       In case the -d option is omitted a command to execute on the\n"     \
  "       specified device can be defined. The software is running as a\n"    \
  "       userspace application, creates the backend specified with the\n"    \
  "       -s option and executes the following command specified with this\n" \
  "       option. The command has to follow the device specification.\n"      \
  "       Otherwise it will be ignored. The command has to be specified for\n"\
  "       each device individually. The following commands are supported:\n"  \
  "       dump : ==> dumps the content of the specified storage\n"            \
  "       ddump: ==> dumps the content of the specified storage with some\n"  \
  "                  additional debug information\n"                          \
  "       erase: ==> invalidates the content of the specified storage\n"      \
  "                  !! Please pay attention here !!\n"                       \
  "                  Don't use this command when the daemon is running and\n" \
  "                  controlling the device which shall be erased because\n"  \
  "                  the daemon will not be informed about the erasure\n"     \
  "                  This leads to a loss of all data written to the\n"       \
  "                  persistent storage after the command was executed.\n\n"  \
  "    -i --info\n\n"                                                         \
  "       Show information about storage\n\n"                                 \
  "    -n --number-rows\n\n"                                                  \
  "       in combination with the dump command (-C --cmd option) only the\n"  \
  "       first n messages will be printed out. Option -d is not allowed\n"   \
  "       in this context (refer to -C --cmd option)\n\n"                     \
  USAGE_DLT                                                                   \
  USAGE_ARCH                                                                  \
  "    -v --verbose\n\n"                                                      \
  "       Displays the command line parameters read by the daemon\n\n"        \
  "    -h --help\n\n"                                                         \
  "       Displays this manual\n\n"                                           \
  "    -m --smoketest\n\n"                                                    \
  "       Executes an internal test functionality\n\n"                        \
  "       Example to create an error memory backend (daemon):\n\n"            \
  "       ./errmemd -d -s /dev/sdk -b 131072 -c 40 -e 1 -w 64 -s /etc/errmem" \
  " -storage -c 100 --bs 4096 --persistent 4 -o\n\n"                          \
  "       This command creates a daemon having /dev/errmem as the frontend\n" \
  "       and two persisten storages.\n"                                      \
  "       The first one and at the same time the default device is a block\n" \
  "       device accessible over /dev/sdk having 40 blocks each of size\n"    \
  "       131072 bytes (128 KB). One block (the first block) will not be\n"   \
  "       overwritten in case all blocks are in use and a wrap around has\n"  \
  "       to be done to store new incoming messages. Write requests to the\n" \
  "       device will be arranged in that way that start and end address\n"   \
  "       are always 64 bytes aligned. The device will be accessed in\n"      \
  "       block mode which means that the daemon assumes that no file \n"     \
  "       system is mounted on this partition. Therfore no create or backup\n"\
  "       functionality is executed.\n\n"                                     \
  "       The second one is a block device having 100 blocks each of size\n"  \
  "       4096 bytes. Four blocks (the first four blocks ) will not be \n"    \
  "       overwritten in case of a wrap around. The storage is accessed\n"    \
  "       in file mode. This means that the daemon creates the storage \n"    \
  "       in case it does not exist and backup it in case it is corrupted\n"  \
  "       or has to be recreated due to new configuration settings.\n"        \
  "       All write requests will be aligned to 4 byte.\n\n"                  \
  "       First example to run errmemd as an application:\n\n"                \
  "       ./errmemd -s /etc/err-stge -C dump -s /dev/sdk -C erase -i\n\n"     \
  "       Runs errmed as an application which first prints out the info of\n" \
  "       both storages defined and then dumps the content of /etc/err-stge\n"\
  "       and invalidates the content of the storage /dev/sdk.\n\n"           \
  "       After executing the commands the application terminates.\n\n"       \
  "       Second example to run errmemd as an application:\n\n"               \
  "       ./errmemd -s /etc/err-stge -C dump -n 50 -s /dev/sdk -C dump -n 30" \
  "       \n\n"                                                               \
  "       Dumps the first fifty messages of storage /etc/err-stge and the\n"  \
  "       first thirty messages of storage /dev/sdk.\n\n"                     \
  "       Example to run the daemon with a bare NOR flash device as\n"        \
  "       a persistent storage:\n\n"                                          \
  "       ./errmemd -d -r /dev/mtd9 -b 131072 -c 8 -e 1 -w 32 -E 0 -R 1\n\n"  \
  "       This command creates a daemon having a bare NOR flash device as\n"  \
  "       persistent storage which has 8 blocks, each with a blocksize\n"     \
  "       of 131072 bytes (128KB). The first block will not be overwritten\n" \
  "       during a life cycle of this persistent storage which ends when\n"   \
  "       the content is erased by command.\n"                                \
  "       When dealing with bare NOR flash devices please take care of the\n" \
  "       following:\n"                                                       \
  "       1.) The blocksize has to match exactly or a multiple of the\n"      \
  "           erasesize of the underlying physical NOR flash device.\n"       \
  "       2.) The write size has to match exactly the write size of the\n"    \
  "           underlying physical NOR flash device. This is the size\n"       \
  "           in bytes for which the device internal flash controller\n"      \
  "           calculates the internal ECC. To set the write size exactly\n"   \
  "           to this size, guarantees that a part of this size of a block\n" \
  "           is programmed only once during the time a block is marked\n"    \
  "           to keep valid data.\n"                                          \
  "       3.) Don't mix up FTL block devices like USB-Stick, SD-Card, etc.\n" \
  "           with bare flash devices. This may lead to errors when the\n"    \
  "           daemon tries to evaluate the device characteristics using\n"    \
  "           IOCTLs which it expects to be supported by that deivce type.\n"

#define MAX_RT_SIG  32

const char* version = "1.3.1 (E19-Release / Requires Driver Version >= 2.10)";
 
// usage - command line help
static void usage()
{
    fprintf(stderr, "\nerrmemd: Daemon to handle messages written into the error memory (RAM).\n" \
            "It automatically reads out the messages stored in RAM and makes them\n" \
            "persistent onto one or more given storage mediums. It provides an socket interface\n" \
            "to give access to this persistent storage in order to make the messages\n" \
            "available for applications for further processing.\n\n"
            "errmemd - Version: %s\n\n", version);
    fprintf(stderr, USAGE);
}

// signal handler

// catch_signal - signal handler
volatile int break_dump = 0;
volatile int terminate  = 0;

static void catch_signal(int signo)
{
    switch (signo) {
    case SIGINT:
    case SIGQUIT:
    case SIGTERM:
        break_dump = 1; // break a possible dump out socket call 
        terminate  = 1; // break main loop and exit daemon execution
		                // or terminate a dumping application
        break;
    case SIGPIPE:
        break;
    case SIGUSR1:
        break;
    case SIGUSR2:
        break;
    default:
        break;
    }

    if (signo >= SIGRTMIN && signo <= SIGRTMAX) {
        if (signo == SIGRTMIN)
            break_dump = 1; // break a possible dump out call
    }
}

// install_signal_handler - install a handler for some signals
static void install_signal_handler()
{
    int signals[] = { SIGINT, SIGQUIT, SIGTERM, SIGPIPE, SIGUSR1, SIGUSR2, 0 };
    int rt_signals[MAX_RT_SIG];
    unsigned i;
    struct sigaction sa;
    struct sigaction rt_sa;

    /* Realtime signals values are mapped to global variable from libc.
     * Therefore the macro values SIGRT* cannot be handled like integer
     * constants 
     */
    rt_signals[0] = SIGRTMIN;
    rt_signals[1] = 0;

    /* Install a signal handler for the above listed standard signals */
    for (i=0; signals[i]; i++) {
        memset(&sa, 0, sizeof(sa));
        sa.sa_handler = catch_signal;
        if (sigaction(signals[i], &sa, NULL)<0)
            syslog(LOG_CRIT, "%s %u", "failed to install signal handler ",
                   signals[i]);
    }

    /* Install a signal handler for the above listed real time signals */
    for (i=0; rt_signals[i]; i++) {
        memset(&rt_sa, 0, sizeof(rt_sa));
        rt_sa.sa_handler = catch_signal;
        if (sigaction(rt_signals[i], &rt_sa, NULL)<0)
            syslog(LOG_CRIT, "%s %u", "failed to install real time signal handler ", signals[i]);
    }   
}

static int32_t errmem_configure_frontend(Errmemd_t *d, uint32_t low, uint32_t high)
{
    int32_t err = 0;
    if (d && d->ifc) {
        if (~high) {
            err = errmem_set_watermark_high(d->ifc, high);
        }
        if (~low) {
            err = errmem_set_watermark_low(d->ifc, low);
        }   
    } else {
        err = -EINVAL;
        syslog(LOG_CRIT, "%s %s",
               "errmem_configure_frontend: Parameter check failed",
               strerror(-err));
    }
    return err;
}

static void errmem_destroy_session(Errmemd_t *d, ErrmemConnect_t** c)
{
    if (d && d->is && c && *c) {
        if (~((*c)->sid)){
            (void)epoll_ctl(d->is->e, EPOLL_CTL_DEL, (*c)->sid, NULL);
            (void)close((*c)->sid);
        }
        if (~(*c)->sinf.fd)
            (void)close((*c)->sinf.fd);
        
        if ((*c)->sinf.block && (~(uintptr_t)(*c)->sinf.block))
            free((*c)->sinf.block);
        
        if ((*c)->sinf.fn && (~(uintptr_t)(*c)->sinf.fn))
            free((*c)->sinf.fn);

        if ((*c)->sinf.msg_queue && (~(uintptr_t)(*c)->sinf.msg_queue)) {
            do {
                MessageQueue_t** mq = errmem_get_message(NULL, &(*c)->sinf.msg_queue);
                if (*mq)
                    free(*mq);
                *mq = NULL;
            } while ((*c)->sinf.msg_queue != NULL);
        }
        free(*c);
        *c = NULL;
    }
}

/* Gets the backend referenced by nr 
 * If nr == 0 the default interface is returned
 */
ErrmemBackend_t* get_backend(Errmemd_t *d, uint32_t nr)
{
    ErrmemBackend_t* cur = NULL;
    int32_t found        = 0;
    if (d && d->backends && ~nr) {
        cur = d->backends;
        while (cur && !found) {
            if (nr > 0) {
                if (cur->get_nr(cur) == nr)
                    found = 1;
            } else if (0 == nr) {
                if (cur->is_default(cur))
                    found = 1;
            }
            if (!found)
                cur = cur->next;
        }
    }
    return cur;
}

static int32_t is_removed_backend(Errmemd_t *d, uint32_t nr)
{
    ErrmemBackend_t* cur = NULL;
    int32_t found        = 0;
    if (d && d->removed && nr > 0) {
        cur = d->removed;
        while (cur && !found) {
            if (cur->get_nr(cur) == nr)
                found = 1;
            else
                cur = cur->next;
        }
    }
    return found;
} 

static void deinit_client(ErrmemConnect_t* c)
{
    if (c) {
        if (~(c->sinf.fd)) {
            (void)close(c->sinf.fd);
            c->sinf.fd = -1;
        }
        if (c->sinf.block && (~((uintptr_t)c->sinf.block)))
            free(c->sinf.block);

        if (c->sinf.msg_queue && (~((uintptr_t)c->sinf.msg_queue))) {
            do {
                MessageQueue_t** mq = errmem_get_message(NULL, &c->sinf.msg_queue);
                if (*mq)
                    free(*mq);
                *mq = NULL;
            } while (c->sinf.msg_queue != NULL);
        }
        c->sinf.backend = NULL;
        c->sinf.block   = NULL;
        c->sinf.status  = ERR_OK;
        c->sinf.state   = ERRMEM_SESSION_ESTABLISHED;       
    }
}

static void init_client(Errmemd_t *d, ErrmemConnect_t* c, int32_t nr_backend)
{
    if (d && d->is && c) {
        c->sinf.status = ERR_OK;
        /* If nr_backend is 0 assign the current default device
         * If nr_backend is > 0 try to assign the backend having that number.
         * In case the default device shall be assigned the session will be
         * created even if no persistent storage is available. The client
         * can activate other output devices like DLT or TRACE.
         * In case a specific backend is requested and this backend is no
         * longer available the session will be kept in the state it is 
         * and error is returned.
         */
        c->sinf.num             = ~0;
        c->sinf.num_start       = ~0;
        c->sinf.pos             =  0;
        c->sinf.seq_num         =  0;
        c->sinf.seq_start       = ~0;
        c->sinf.block_cur       = ~0;
        c->sinf.seq_max         =  0;
        c->sinf.block_magic     = ~0;
        c->sinf.seq_last        = ~0;
        c->sinf.seq_next        = ~0;
        c->sinf.seq_new         = ~0;
        c->sinf.persistent      =  0;
        c->sinf.client_mode     =  0; /* this is socket server action */
        c->sinf.block           = NULL;
        c->sinf.msg_queue       = NULL;
        c->sinf.fn              = NULL;

        c->sinf.backend = get_backend(d, nr_backend);

        if (c->sinf.backend) {
            c->sinf.fd = -1;
            c->sinf.backend->open_session(&c->sinf);
            if (c->sinf.status < 0)
                syslog(LOG_CRIT, "[%d]: %s %d %s %s",
                       d->is->con.sid,
                       "init client: cannot open backend for cliend id: ",
                       c->sid, " - error = ", strerror(c->sinf.status));
        } else if (nr_backend > 0) {
            if (is_removed_backend(d, nr_backend)) {
                c->sinf.status = -ECANCELED;
                syslog(LOG_CRIT, "[%d]: %s %d %s %d %s %s",
                       d->is->con.sid,
                       "init client: requested backend: ", nr_backend,
                       " for client session id: ", c->sid,
                       " is not available - error = ",
                       strerror(c->sinf.status));
            } else {
                c->sinf.status = -ENODEV;
                syslog(LOG_CRIT, "[%d]: %s %d %s %d %s %s",
                       d->is->con.sid,
                       "init client: requested backend: ", nr_backend,
                       " for client session id: ", c->sid,
                       " does not exist - error = ",
                       strerror(c->sinf.status));
            }       
        }
    } else if (c) {
        c->sinf.status = -EINVAL;
        syslog(LOG_CRIT, "%s:", "init client: Parameter check failed");
    } else
        syslog(LOG_CRIT, "%s:", "init client: Client structure is NULL");
}

static void errmem_recv_socket_data(void *p, int32_t len, ErrmemConnect_t *c)
{
    char* data    = (char*)p;
    int32_t rcv   = 0;
    int32_t rc    = 0;
    int32_t again = 1;

    /* check for data on the socket */
    if (p && c && len > 0) {
        while (again) {
            rc = recv(c->sid, data, len, 0);
            if (rc < 0 ) {
                if (errno != EAGAIN && errno != EWOULDBLOCK) {
                    c->sinf.status = -errno;
                    syslog(LOG_CRIT, "[%d] %s %s", c->sid,
                           "errmem_recv_socket_data: failed receiving data "
                           "- errno:", strerror(-errno));
                    again = 0;
                }
            } else if (rc == 0) {
                c->sinf.status = -EPIPE;                                    
                again = 0;
            } else if (rc < len){
                rcv += rc;
                if (rcv == len){
                    again = 0;
                    c->sinf.status = ERR_OK;                    
                } else if (rcv > len){
                    again = 0;
                    c->sinf.status = -EMSGSIZE;
                } else {
                    data += rc;
                    c->sinf.status = -EAGAIN;
                }
                syslog(LOG_INFO, "[%d] %s %d %d", c->sid, 
                       "errmem_recv_socket_data: requested / received: ",
                       len, rcv);
            }
            else if (rc == len) {
                again = 0;
                c->sinf.status = ERR_OK;
            } else {
                c->sinf.status = -EMSGSIZE;
            }
        }
        if (c->sinf.status < 0)
            syslog(LOG_INFO, "[%d] %s %d %s %d %s %s",c->sid,
                   "errmem_recv_socket_data: received", rcv,
                   "bytes - requested:", len, "error = ", 
                   strerror(c->sinf.status));
    }
    else {
        syslog(LOG_CRIT, "%s",
               "errmem_recv_socket_data: NULL Pointer passed to function");
    }
}

static int32_t create_backend_structure(Errmemd_t* ed, ErrmemBackendDev_t* dev)
{
    int32_t err = 0;
    ErrmemBackend_t*  b   = NULL;
    ErrmemBackend_t** cur = NULL;
    
    if (ed && dev) {
        b = errmem_backend_factory(dev);
        if (b) {
            if ((b->get_type) && (b->get_type(b) & ERRMEM_ALL_OUT))
                cur = &ed->outputs;
            else
                cur = &ed->backends;
            
            err = errmem_add_backend(cur, b);
        }
    }
    return err;
}

static void reinit_session(Errmemd_t *d, ErrmemConnect_t *c)
{
    int32_t cb = 0;
    if (c->sinf.backend && c->sinf.backend->get_nr)
        cb = c->sinf.backend->get_nr(c->sinf.backend);
    deinit_client(c);
    init_client(d, c, cb);
}

static int32_t errmem_handle_client(Errmemd_t *d, ErrmemConnect_t *c)
{
    struct msg_header  pac     = {.type = MSG_REQUEST_UNKNOWN, .status = 0, .len = 0};
    struct msg_header *p       = &pac;
    void              *buf_in  = NULL;
    void              *buf_out = NULL;
    int32_t            rc      =  0;
    int32_t            sid     = -1;
    MessageQueue_t**   mq      = NULL;
    
    if (d && c){
        /* read message header from the socket */
        errmem_recv_socket_data(p, sizeof(*p), c);
        if (ERR_OK == c->sinf.status) {
            /* if additional data available read them */
            if (p->len > sizeof(*p)) {
                buf_in  = calloc(1, (p->len - sizeof(*p)));
                if (NULL == buf_in) {
                    syslog(LOG_CRIT, "[%d]: %s %d",
                           ((ErrmemConnect_t*)(d->ifc->con))->sid, 
                           "errmem_handle_client: Failed to allocate message "
                           "buffer for payload from client: ", c->sid);
                    c->sinf.status = -ENOMEM;
                } else
                    errmem_recv_socket_data(buf_in, (p->len - sizeof(*p)), c);
            }
            /* Check the backend state according to requested action.
             * Read and Dump functionality are related to the assigned
             * persistent storage only
             */
            if (p->type == MSG_REQUEST_READ  || 
                p->type == MSG_REQUEST_DUMP) {
                if (d->stop)
                    c->sinf.status = -ENOEXEC;
                else if (c->sinf.backend) {
                    if (c->sinf.backend->get_nr) {
                        uint32_t nr = c->sinf.backend->get_nr(c->sinf.backend);
                        if (is_removed_backend(d, nr)) {
                            c->sinf.status = -EIDRM;
                            syslog(LOG_CRIT, "[%d]: %s %d",
                                   d->is->con.sid,
                                   "errmem_handle_client: Backend "
                                   "not accessible - session id: ",
                                   c->sid);
                        } else if (!(get_backend(d, nr))) {
                            c->sinf.status = -ENOMEDIUM;
                            syslog(LOG_CRIT, "[%d]: %s %d",
                                   d->is->con.sid,
                                   "errmem_handle_client: Backend "
                                   "not in backend list - session id: ",
                                   c->sid);
                        }
                    } else {
                        c->sinf.status = -ENODEV;
                        syslog(LOG_CRIT, "[%d]: %s %d",
                               d->is->con.sid,
                               "errmem_handle_client: Can't get "
                               "number of backend - session id: ",
                               c->sid);
                    }
                } else {
                    c->sinf.status = -EPERM;
                    syslog(LOG_CRIT, "[%d]: %s %d",
                           d->is->con.sid,
                           "errmem_handle_client: No backend assigned. "
                           "Read/Dump not possible - session id: ", c->sid);
                }
            } else if (p->type == MSG_REQUEST_SET_STORAGE) {
                /* This can touch arbitrary devices which have nothing
                 * to do with the current session
                 */
                if (d->stop)
                    c->sinf.status = -ENOEXEC;
                else if (is_removed_backend(d, p->status)) {
                    /* In case the default backend is requested it can't 
                     * be found in the remove queue. We don't have to
                     * do sparated checks for default device here.
                     */
                    c->sinf.status = -EIDRM;
                    syslog(LOG_CRIT, "[%d]: %s %d",
                           d->is->con.sid,
                           "errmem_handle_client: Backend "
                           "not accessible - session id: ",
                           c->sid);
                } else if (!(get_backend(d, p->status))) {
                    c->sinf.status = -ENOMEDIUM;
                    syslog(LOG_CRIT, "[%d]: %s %d",
                           d->is->con.sid,
                           "errmem_handle_client: Backend "
                           "not in backend list - session id: ",
                           c->sid);
                }
            } else if (p->type == MSG_REQUEST_ERASE) {
                /* This can touch arbitrary devices which have nothing
                 * to do with the current session
                 */
                if (d->stop)
                    c->sinf.status = -ENOEXEC;
                else if (!p->status) {
                    if (!d->backends) {
                        c->sinf.status = -EPERM;
                        syslog(LOG_CRIT, "[%d]: %s %d",
                               d->is->con.sid,
                               "errmem_handle_client: No persistent storages "
                               "accessible. Erase not possible - session id: ",
                               c->sid);
                    }
                } else if (p->status > 0) {
                    if (is_removed_backend(d, p->status)) {
                    /* In case the default backend is requested it can't 
                     * be found in the remove queue. We don't have to
                     * do sparated checks for default device here.
                     */
                        c->sinf.status = -EIDRM;
                        syslog(LOG_CRIT, "[%d]: %s %d",
                               d->is->con.sid,
                               "errmem_handle_client: Backend "
                               "not accessible - session id: ",
                               c->sid);
                    } else if (!(get_backend(d, p->status))) {
                        c->sinf.status = -ENOMEDIUM;
                        syslog(LOG_CRIT, "[%d]: %s %d",
                               d->is->con.sid,
                               "errmem_handle_client: Backend "
                               "not in backend list - session id: ",
                               c->sid);
                    }
                } else {
                    c->sinf.status = -EINVAL;
                    syslog(LOG_CRIT, "[%d]: %s %d",
                           d->is->con.sid,
                           "errmem_handle_client: Invalid "
                           "backend id to erase requested  - session id: ",
                           c->sid);             
                }
            }

            if (ERR_OK == c->sinf.status) {
                switch (p->type){
                case MSG_REQUEST_READ:
                    if ((c->sinf.state == ERRMEM_SESSION_READ_DONE) &&
                             (!c->sinf.msg_queue))
                        c->sinf.status = ERR_READ_END;
                    else if (c->sinf.state == ERRMEM_SESSION_BACKEND_ERASED &&
                             (!c->sinf.msg_queue))
                        c->sinf.status = ERR_SESSION_BACKEND_ERASED;
                    if (c->sinf.status == ERR_OK){
                        if (c->sinf.backend) {
                            if (!c->sinf.msg_queue) {
                                c->sinf.msg_queue = calloc(1, sizeof(MessageQueue_t));
                                if (c->sinf.msg_queue) {
                                    c->sinf.backend->read_session(&c->sinf);
                                    errmem_create_messages(&c->sinf);
                                }
                            }
                            if (c->sinf.state == ERRMEM_SESSION_READ_DONE) {
                                if (c->sinf.msg_queue && !(c->sinf.msg_queue->next)) 
                                    p->status = ERR_READ_END;
                                else
                                    p->status = ERR_OK;
                            } else if (c->sinf.state == ERRMEM_SESSION_BACKEND_ERASED)
                                p->status = ERR_SESSION_BACKEND_ERASED;
                            else if (c->sinf.status < 0)
                                p->status = c->sinf.status;
                            else
                                p->status = ERR_OK;

                            mq = errmem_get_message(p, &c->sinf.msg_queue);
                            if (*mq) {
                                buf_out = calloc(1, p->len);

                                if (!buf_out){
                                    c->sinf.status = -ENOMEM;
                                    syslog(LOG_CRIT, "[%d] %s %m", c->sid, "handle client: no memory for send buffer");
                                }
                                else {
                                    memcpy(buf_out, p, sizeof(*p));
                                    memcpy((addptr(buf_out, sizeof(*p))), (void*)&(*mq)->msg, (p->len - sizeof(*p)));
                                }
                            }
                            if (mq && *mq) {
                                free(*mq);
                                *mq = NULL;
                                mq  = NULL;
                            }
                        } else {
                            syslog(LOG_CRIT, "[%d] %s %m", c->sid, "backend is going lost during read request");
                            c->sinf.status = -ENOTRECOVERABLE;
                        }
                    }
                    break;
                case MSG_REQUEST_SET_STORAGE:
                    deinit_client(c);
                    if (!c->sinf.status && p->status >= 0)
                        init_client(d, c, p->status);
                    else
                        c->sinf.status = -EINVAL;
                    break;
                case MSG_REQUEST_ERASE:
                    if (!p->status)
                        errmem_backend_erase_content(d->backends);
                    else if (p->status > 0) {
                        ErrmemBackend_t* backend = get_backend(d, p->status);
                        if (backend && backend->erase) {
                            c->sinf.status = backend->erase(backend);
                        } else
                            c->sinf.status = -ENOMEDIUM;
                    } else
                        c->sinf.status = -EINVAL;
                    break;
                case MSG_REQUEST_STOP:
                    d->stop = 1;
                    d->p_mode = 0;
                    if (d->ifc->con && !(d->outputs))
                        c->sinf.status = errmem_disconnect_frontend(d);
                    break;
                case MSG_REQUEST_DUMP:
                    if (c->sinf.state != ERRMEM_SESSION_ESTABLISHED)
                        c->sinf.status = -EBUSY;
                    else {
                        ErrmemBackendDev_t eb = {0};
                        ErrmemBackend_t* backend = NULL;
                        eb.type = p->status & ERRMEM_ALL_OUT;
                        backend = errmem_backend_factory(&eb);                      
                        if (!backend)
                            c->sinf.status = -EACCES;
                        else {
                            c->sinf.backend->dump(NULL, backend, &c->sinf, NULL, 0, 0);
                            c->sinf.status = ERR_OK;
                            if (backend->destroy)
                                backend->destroy(backend);
                        }
                    }
                    break;
                case MSG_REQUEST_OUT_DEV:
                    if (!(p->status & ERRMEM_ALL_OUT))
                        c->sinf.status = -EINVAL;
                    else {
                        int32_t found = 0;
                        int32_t err = 0;
                        /* non persistent output devices already defined */
                        if (d->outputs) {
                            /* check whether requested one is already in queue */
                            ErrmemBackend_t* out = d->outputs;
                            while (out && !found) {
                                if (out->get_type &&
                                    (out->get_type(out) == 
                                     (uint32_t)(p->status & ERRMEM_ALL_OUT)))
                                    found = 1;
                                else
                                    out = out->next;
                            }
                        }
                        if (!found) {
                            /* create pure output backend */
                            ErrmemBackendDev_t eb = {0};
                            /* for type eliminate possible attributes */
                            eb.type = (p->status & ERRMEM_ALL_OUT);
                            err = create_backend_structure(d, &eb);
                            if (err < 0)
                                c->sinf.status = err;
                        }
                        if (c->sinf.status == ERR_OK) {
                            /* Settings for continuous or once only output
                             * as well as setting for acknowledge or not
                             * acknowledge messages to the frontend are stored */
                            if (p->status & ERRMEM_DUMP_ONCE)
                                d->out_once = 1;
                            else
                                d->out_once = 0;
                            if (p->status & ERRMEM_ACK_MSG)
                                d->out_ack = 1;
                            else
                                d->out_ack = 0;
                            /* check connection to frontend in case we have to
                             * do something here */
                            if (!d->p_mode && d->outputs) {
                                /* reconnect if not connected */
                                if (!d->ifc->con)
                                    c->sinf.status = errmem_connect_frontend(d);
                                /* if dump shall be an one-time do it here */
                                if (c->sinf.status >= 0 && d->out_once) {
                                    do {
                                        err = errmem_handle_frontend(d, NULL);
                                    } while (err > 0);
                                    c->sinf.status = err;
                                    if (d->ifc->con)
                                        err = errmem_disconnect_frontend(d);
                                }
                            }
                        }
                    }
                    break;
                default:
                    c->sinf.status = -ENOSYS;
                    break;
                }
            }
        }
        sid = c->sid;
        p->status = c->sinf.status;
        if (buf_out)
            rc = send(sid, buf_out, p->len, 0);
        else {
            rc = send(sid, p, sizeof(*p), 0);
        }
        if (rc != (int32_t)(p->len) && rc != sizeof(*p)){
            syslog(LOG_CRIT, "[%d] %s %m", sid, 
                   "handle client: failed to send answer");
            c->sinf.status = -EIO;
        }

        /* If a read session is over reinitialize the session context */
        if (c->sinf.status == ERR_READ_END ||
            c->sinf.status == ERR_SESSION_BACKEND_ERASED) {
            reinit_session(d, c);
        }
    } else {
        syslog(LOG_CRIT, "%s %m", "handle client: null Pointer");
        rc = -1;
    }
    
    if (buf_in)
        free(buf_in);
    
    if (buf_out)
        free(buf_out);
    
    return rc;
}

static ErrmemConnect_t* create_client(ErrmemSocket_t *s, int32_t sid)
{
    int32_t err            = 0;
    struct epoll_event ev  = {.events = EPOLLIN};
    ErrmemConnect_t *c   = malloc(sizeof(*c));
    
    if (!c) {
        syslog(LOG_CRIT, "%s %s",
               "create client: malloc for client connection context failed"
               " - error: ", strerror(ENOMEM));
    } else {
        memset(c, 0xFF, sizeof(*c));
        c->sid         = sid;
        c->h           = errmem_handle_client;
        c->sinf.status = ERR_OK;
        err = fcntl(c->sid, F_SETFL, O_NONBLOCK);
        if (-1 == err) {
            err = -errno;
            syslog(LOG_CRIT, "%s %s",
                   "create client: fcntl F_SETFL failed - error = ",
                   strerror(-err));
        } else {
            ev.data.ptr = c;
            err = epoll_ctl(s->e, EPOLL_CTL_ADD, c->sid, &ev);
            if (-1 == err){
                err = -errno;
                syslog(LOG_CRIT, "%s %s",
                       "create client: client connection cannot be added to "
                       "poll set - error = ", strerror(-err));
            }
        }
        if (err < 0)
            c->sinf.status = err;
    }
    return c;
}

/* This function handles the server socket of the daemon itself .
 * All incoming registrations requests of clients end up here.
 * Note: Even in case the daemon is stopped a session will be created.
 * This is because an output device can be set.
 */
static int32_t errmem_handle_server_socket(Errmemd_t *d, ErrmemConnect_t* c)
{
    int32_t err         = ERR_OK;
    int32_t sdata       = 0;
    ErrmemConnect_t* cc = NULL;
    struct msg_header p = {.type = MSG_REQUEST_ESTABLISH, .len = sizeof(p)};
    int32_t sid         = -1;
    
    if (d && c) {
        sid = accept(c->sid, NULL, NULL);
        if (-1 == sid) {
            err = -errno;
            syslog(LOG_CRIT, "[%d]: %s %s",
                   c->sid, "accept failed on server socket - error = ", strerror(-err));
        } else {
            cc = create_client(d->is, sid);
            if (cc) {
                if (cc->sinf.status == ERR_OK) {
                    init_client(d, cc, 0);
                    if (cc->sinf.status == ERR_OK)
                        cc->sinf.state  = ERRMEM_SESSION_ESTABLISHED;
                    else
                        err = cc->sinf.status;
                } else
                    err = cc->sinf.status;
            } else
                err = -ENOMEM;
        }
        /* answer client and provide status of session */
        p.status = err;
        sdata = send(sid, &p, p.len, 0);
        if (sdata < 0) {
            err = -errno;
            syslog(LOG_CRIT, "[%d]: %s %d %s %s", c->sid, 
                   "Failed to send data to client with socket id: ", sid,
                   " error = ", strerror(-err));
        }

        if (err != ERR_OK) {
            errmem_destroy_session(d, &cc);
            syslog(LOG_INFO, "[%d]: %s %d %s %s",
                   c->sid, "create client: client session on id: ", sid,
                   " has been destroyed - error = ", strerror(-err));
        }
    } else {
        err = -EINVAL;
        syslog(LOG_CRIT, "%s", "create client: Parameter check failed");
    }
    return err;
}

static void errmem_socket_destroy(ErrmemSocket_t* s) 
{
    if (s) {
        if (~s->con.sid){
            int32_t err = epoll_ctl(s->e, EPOLL_CTL_DEL, s->con.sid, NULL);
            if (-1 == err)
                syslog(LOG_CRIT, "%s: %m [%d]", "cannot delete server from poll set", s->con.sid);
            (void)close(s->con.sid);
        }
        if (~s->e)
            close(s->e);
        free(s);
        s = NULL;
    }
}

static void errmemd_destroy(Errmemd_t* errmemd)
{
    if (errmemd) {
        errmem_destroy(errmemd->ifc);
        errmem_socket_destroy(errmemd->is);
        errmem_backend_destroy(errmemd->backends);
        errmem_backend_destroy(errmemd->outputs);
        errmem_backend_destroy(errmemd->removed);

        if (errmemd->tcpip_address)
            free(errmemd->tcpip_address);
        if (errmemd->fill_path)
            free(errmemd->fill_path);
        if (errmemd->frontend)
            free(errmemd->frontend);
    }
}

static void create_backends(Errmemd_t* ed)
{
    int32_t access_mode = 0;
#ifdef _ARCH_x86
    if (ed->type == ERRMEM_X86_NO_DEAMON ||
        ed->type == ERRMEM_X86)
#else
    if (ed->type == ERRMEM_DAEMON)
#endif
        access_mode |= O_RDWR;
    else
        access_mode |= O_RDONLY;
    
    errmem_backend_create(access_mode, &(ed->backends));
        
    errmem_backend_create(access_mode, &(ed->outputs));

    if (ed->backends)
        ed->p_mode = 1;
}

#ifdef _CHECK_PROCESS_
static pid_t check_process(const char* prc)
{
    pid_t   pid  = 0;
    int32_t sz   = 50;
    char buf[sz];
    FILE* check  = NULL;
    char* res    = NULL;
    char* cur    = NULL;
    pid_t own  = getpid();
    if (snprintf(buf, sz, "pidof %s", prc) >= sz) {
        syslog(LOG_CRIT, "%s %s", "not enough memory allocated to "
               "check for an already running daemon", prc);
        pid = -1;
    }
    if (!pid) {
        check = popen(buf, "r");
        if (check) {
            char* r = fgets(buf, sz, check);
            if (r && (r == buf)) {
                cur = &buf[0];
                do {
                    pid = strtoul(cur, &res, 10);
                    if (pid && (pid != own))
                        break;
                    cur = res;
                } while (pid != 0 && cur < ((char*)buf + sz));
            } else {
                syslog(LOG_CRIT, "%s %s", "failed to get result from popen", prc);
                pid = -1;
            }
            pclose(check);
        } else {
            syslog(LOG_CRIT, "%s %s %s", "faild to execute command ", prc, buf);
            pid = -1;
        }
    }
    return pid;
}
#endif

static char* check_tcpip_address(const char* s)
{
    int32_t err = 0;
    char*   in  = 0;

    if (s) 
    {
        err = strlen(s);
        in  = strdup(s);

        if (err < 7 || err > 15){
            err = -1;
            syslog(LOG_CRIT, "%s %s", "tcpip address (-a) is invalid (4 bytes "
                    "separated by dot, e.g. ddd.ddd.ddd.ddd or dd.d.ddd.d) "
                    " - check format of ", s);
        }
        if (!in) {
            syslog(LOG_CRIT, "%s %s", "cannot allocate to duplicate ", s);
            err = -1;
        }
        if (~err) {
            char* p = strtok(in, ".");
            if (p != in) {
                syslog(LOG_CRIT, "%s %s", "tcpip address (-a) is not allowed to start with '.'", s);
                err = -1;
            }
            if (!p) {
                syslog(LOG_CRIT, "%s %s", "bytes in tcpip address not separated by '.'", s);
                err = -1;
            }
            if (~err) {
                int32_t ct = 0;
                while (p) {
                    char* end = 0;
                    errno = 0;
                    int32_t b = strtol(p , &end, 10);
                    if (b < 0 ) {
                        syslog(LOG_CRIT, "%s %s", "byte is negativ: ", p);
                        err = -1;
                        break;
                    }
                    if (b > 255) {
                        syslog(LOG_CRIT, "%s %d", "invalid value greater than "
                               " 255 (1 byte)", b);
                        err = -1;
                        break;                  
                    }
                    if ((end == p) || (*end != '\0')) {
                        syslog(LOG_CRIT, "%s %s", "value is not a valid part of a "
                                "tcpip address: ", p);
                        err = -1;
                        break;                                      
                    }
                    if (errno) {
                        syslog(LOG_CRIT, "%s %s %m", "byte in tcpip address is not valid: ", p);
                        err = -1;
                        break;
                    } else {
                        ++ct;
                        p = strtok(NULL,".");
                    }
                }
                if (~err && ct == 4)
                    err = 0;
                else
                    syslog(LOG_CRIT, "%s %s", "tcpip address is not valid: ", s);
            }
        }
        if (in)
            free(in);
        
        if (!err)
            in = strdup(s);
        else
            in = NULL;
    }
    return in;  
}

int32_t check_option_arg(int32_t popt, const char* arg,
                         const char* so, const struct option* lo)
{
    int32_t  ret     = -1;
    int32_t  len     = 0;
    uint32_t has_arg = 0;

    if (so && lo) {
        if (popt == -1 || popt == '?' || popt == ':')
            ret = popt;
        else {
            if (popt >= LONGOPT_BASE && popt <= LONGOPT_END)
                has_arg = lo[popt - LONGOPT_BASE].has_arg;
            else
                has_arg = (uintptr_t)strchr(so, popt);
        }

        if (has_arg) {
            int32_t tmp = 0;
            char*   end = 0;
            switch(popt) {
            case  's':
            case  'r':
            case  'f':
            case  'y':
            case  'z':
            case  'C':
            case  LONGOPT_BASE+10:
            case  LONGOPT_BASE+16:
            case  LONGOPT_BASE+23:
            case  LONGOPT_BASE+26:
            case  LONGOPT_BASE+27:
                /* parameter is mandatory */
                ret = popt;
                if (arg) {
                    len = strlen(arg);
                    if (!len) {
                        ret = ':';
                        syslog(LOG_CRIT, "%s %s %s", "option ", (char*)&popt ,
                               "has no argument");
                    } else if ((*arg) == '-') {
                        ret = ':';
                        syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt,
                               "has invalid argument: ", arg);
                    } else if (isdigit(*arg)) {
                        syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt, 
                               "has invalid argument: ", arg);
                        ret = ':';
                    }
                } else {
                    syslog(LOG_CRIT, "%s %s %s", "option ", (char*)&popt ,
                           "has no argument");
                    ret = ':';
                }
                break;
            case  'm':
            case  'd':
            case  LONGOPT_BASE+3:
            case  LONGOPT_BASE+4:
                /* have an optional charcter parameter */
                ret = popt;
                if (arg) {
                    len = strlen(arg);
                    if (len) {
                        if ((*arg) == '-')
                            ret = ':';
                        else if (isdigit(*arg)) {
                            syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt,
                                   "has invalid argument: ", arg);
                            ret = ':';
                        }
                    }
                }
                break;
            case  'o':
                /* has an optional integer parameter */
                ret = popt;
                if (arg){
                    errno = 0;
                    tmp = strtol(arg, &end, 0);
                    if (errno) {
                        ret = ':';
                        syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt,
                               " has invalid argument: ", arg);
                    } else if (tmp < 0) {
                        ret = ':';
                        syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt,
                               " has invalid negative argument: ", arg);
                    } else if (tmp >= OPT_ACC_END) {
                        ret = ':';
                        syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt,
                               " has no such access option defined: ", arg);
                    } else if (*end != '\0') {
                        ret = ':';
                        syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt, 
                               " has invalid argument: ", arg);
                    }
                }
                break;
            case  'u':
            case  'l':
            case  'p':
            case  'b':
            case  'c':
            case  'e':
            case  'w':
            case  'E':
            case  'R':
            case  LONGOPT_BASE+5:
            case  LONGOPT_BASE+6:
            case  LONGOPT_BASE+8:
            case  LONGOPT_BASE+12:
            case  LONGOPT_BASE+13:
            case  LONGOPT_BASE+14:
            case  LONGOPT_BASE+15:
            case  LONGOPT_BASE+17:
            case  LONGOPT_BASE+18:
            case  LONGOPT_BASE+19:
            case  LONGOPT_BASE+29:
            case  LONGOPT_BASE+30:
                errno = 0;
                tmp = strtol(arg, &end, 0);
                if (errno) {
                    ret = ':';
                    syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt,
                           " has invalid argument: ", arg);
                } else if (tmp < 0) {
                    ret = ':';
                    syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt,
                           " has invalid negative argument: ", arg);
                } else if (*end != '\0') {
                    ret = ':';
                    syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt, 
                           " has invalid argument: ", arg);
                } else
                    ret = popt;
                break;
            case  'n':
            case  LONGOPT_BASE+25:
                errno = 0;
                strtol(arg, &end, 0);
                if (errno) {
                    ret = ':';
                    syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt,
                           "has invalid argument: ", arg);
                } else if (*end != '\0') {
                    ret = ':';
                    syslog(LOG_CRIT, "%s %s %s %s", "option ", (char*)&popt,
                           "has invalid argument: ", arg);
                } else
                    ret = popt;
                break;
            default:
                ret = popt;
                break;
            }
        } else
            ret = popt;
    } else
        ret = popt;     
    return ret;
}

static int32_t errmem_create_socket(ErrmemSocket_t *s)
{
    int32_t err = socket(PF_UNIX, SOCK_STREAM, 0);
    if (-1 != err) {
        struct sockaddr_un addr   = {
            .sun_family = AF_UNIX
        };
        memcpy(addr.sun_path, SOCKET_PATH, sizeof(SOCKET_PATH));
        s->con.sid = err;
        err = bind(s->con.sid, (struct sockaddr *)&addr,
                   __builtin_offsetof(struct sockaddr_un, sun_path)
                   + sizeof(SOCKET_NAME));
        if (-1 == err) {
            err = -errno;
            syslog(LOG_CRIT, "%s %s", "bind to socket failed - error = ", strerror(-err));
        }
    } else {
        err = -errno;
        syslog(LOG_CRIT, "%s %s", "socket call failed - error = ", strerror(-err));
    }

    if (0 <= err) {
        err = listen(s->con.sid, 1);
        if (-1 == err) {
            err = -errno;
            syslog(LOG_CRIT, "%s %s", "listen call on socket failed - error = ", strerror(-err));
        }
    }
    return err;
}
    
static int32_t errmem_create_server(ErrmemSocket_t* s)
{
    int32_t            err = 0;
    struct epoll_event ev  = {.events = EPOLLIN};

    /* set server handler for incoming socket requests */   
    s->con.h = errmem_handle_server_socket;
    
    err = errmem_create_socket(s);

    if(0 <= err) {
        err = epoll_create(EXPECTED);
        if (-1 == err) {
            err = -errno;
            syslog(LOG_CRIT, "%s %s", "server poll set not created - error = " , strerror(-err));
        } else {
            s->e = err;
            ev.data.ptr = &s->con;
            err = epoll_ctl(s->e, EPOLL_CTL_ADD, s->con.sid, &ev);
            if (-1 == err) {
                err = -errno;
                syslog(LOG_CRIT, "%s %s", "server socket not added to poll set - error = ", strerror(-err));
            }
        }
    }
    return err;
}

static void process(Errmemd_t *d)
{
    int32_t err = 0;
    sd_notify(0, "READY=1");
    while (d && d->is && !terminate) {
        struct epoll_event events = {0};
        ErrmemConnect_t* c        = NULL;
        err = epoll_wait(d->is->e, &events, 1, -1);
        c = events.data.ptr;
        if (((-1 == err) && (EINTR != errno)) ||
            events.events & EPOLLERR          ||
            events.events & EPOLLHUP          ||
            !(events.events & EPOLLIN)) {
            if (c) {
                syslog(LOG_INFO, "[%d]: %s %d", d->is->con.sid,
                       "Close client connection: ", c->sid);
                errmem_destroy_session(d, &c);
            }
        } else if ((-1 == err) && (EINTR == errno))
            continue;
        else if (c)
            err = c->h(d, c);
        else
            continue;
    }
}

static void errmem_smoketest(char* fe)
{
    ErrmemInterface_t* hdl_errmem = errmem_create(fe);
    if (hdl_errmem)
        errmem_info(hdl_errmem, stdout);
    errmem_destroy(hdl_errmem);
}

int main (int argc, char* argv[])
{
    int32_t    err     = 0;
    int32_t    opt     = 0;
    int32_t    opt_idx = 0;
    uint32_t   repeat  = 0;
    uint32_t   nr      = 0;

    int32_t    info    = 0;
    int32_t    verbose = 0;

    uint32_t   w_high  = ~0;
    uint32_t   w_low   = ~0;
    int32_t    nr_dump =  0;
    char*      filter  = NULL;
    Errmemd_t* errmemd = NULL;
    
    const char* const sh_opt     = ":hivm::d::u:l:a:p:ks:o::b:c:e:w:f:gxy:n:z:C:r:E:R:O";
    const char* const sh_opt_arg = "mdulapsobcewfynzCrER";

    const struct option long_opt[] = {
        { "help",           no_argument,       NULL, LONGOPT_BASE+0}, /* h */
        { "info",           no_argument,       NULL, LONGOPT_BASE+1}, /* i */
        { "verbose",        no_argument,       NULL, LONGOPT_BASE+2}, /* v */
        { "smoketest",      optional_argument, NULL, LONGOPT_BASE+3}, /* m */
        { "daemon",         optional_argument, NULL, LONGOPT_BASE+4}, /* d */
        { "watermark-high", required_argument, NULL, LONGOPT_BASE+5}, /* u */
        { "watermark-low",  required_argument, NULL, LONGOPT_BASE+6}, /* l */
        { "tcpip-address",  required_argument, NULL, LONGOPT_BASE+7}, /* a */
        { "tcpip-port",     required_argument, NULL, LONGOPT_BASE+8}, /* p */
        { "ack-msg-to-drv", no_argument,       NULL, LONGOPT_BASE+9}, /* k */
        { "block-storage",  required_argument, NULL, LONGOPT_BASE+10},/* s */
        { "access_options", optional_argument, NULL, LONGOPT_BASE+11},/* o */
        { "bs",             required_argument, NULL, LONGOPT_BASE+12},/* b */
        { "cnt",            required_argument, NULL, LONGOPT_BASE+13},/* c */
        { "persistent",     required_argument, NULL, LONGOPT_BASE+14},/* e */
        { "write-size",     required_argument, NULL, LONGOPT_BASE+15},/* w */
        { "file-storage",   required_argument, NULL, LONGOPT_BASE+16},/* f */
        { "file-size",      required_argument, NULL, LONGOPT_BASE+17},/* b */
        { "file-cnt",       required_argument, NULL, LONGOPT_BASE+18},/* c */
        { "file-persistent",required_argument, NULL, LONGOPT_BASE+19},/* e */
        { "dlt-output",     no_argument,       NULL, LONGOPT_BASE+21},/* g */
        { "x86_ndaemon_dbg",no_argument,       NULL, LONGOPT_BASE+22},/* x */
        { "x86_fill_source",required_argument, NULL, LONGOPT_BASE+23},/* y */
        { "number-rows",    required_argument, NULL, LONGOPT_BASE+25},/* n */
        { "filtered",       required_argument, NULL, LONGOPT_BASE+26},/* z */
        { "cmd",            required_argument, NULL, LONGOPT_BASE+27},/* C */
        { "flash-storage",  required_argument, NULL, LONGOPT_BASE+28},/* r */
        { "erase-wait-time",required_argument, NULL, LONGOPT_BASE+29},/* E */
        { "erase-retry",    required_argument, NULL, LONGOPT_BASE+30},/* R */
        { "once",           no_argument,       NULL, LONGOPT_BASE+31},/* O */
        { NULL,             0,                 NULL,  0  } /* Required at end of array.  */
    };

    char p[] = "errmemd[XXXXXXXXXX]";
    (void)snprintf(p, sizeof(p), "errmemd[%d]", getpid());
    openlog(p, LOG_CONS | LOG_PERROR, 0);

    errmemd = calloc(1, sizeof(Errmemd_t));
    if (!errmemd)
        syslog(LOG_CRIT, "%s", "can't allocate memory for application");

    while (errmemd) {
        int32_t s_opt = 0;
        ErrmemBackendDev_t eb = {0};
        eb.persistent = ~0;
        opterr = 0;
        /* if an option read in an inner loop was not meant to be handled there */
        if (!repeat)
            opt = getopt_long(argc, argv, sh_opt, long_opt, &opt_idx);
        repeat = 0;
        if (opt == -1)
            break;

        /* check option argument because getopt does not do it */
        if (opt != ':' && opt != '?') {
            s_opt = opt;
            opt = check_option_arg(opt, optarg, sh_opt_arg, long_opt);
            if (opt == ':')
                optopt = s_opt;
        }
        switch(opt) {
        case 'h':
        case LONGOPT_BASE+0:
            usage();
            if (argc > 2)
                syslog(LOG_INFO, "%s", "check command line parameters -  "
                       "option help was executed, others ignored");
            err = -1;
            break;
        case 'i':
        case LONGOPT_BASE+1:
            info = 1;
            break;
        case 'v':
        case LONGOPT_BASE+2:
            if (verbose) {
                syslog(LOG_CRIT, "%s", "check command line parameters - "
                       "option -v (--verbose) specified twice");
                err = -1;
                break;
            }
            verbose = 1;
            break;
        case 'm':
        case LONGOPT_BASE+3:
            errmem_smoketest(optarg);
            if (argc > 3)
                syslog(LOG_CRIT, "%s", "check command line parameters - "
                       "option smoketest was executed, others ignored");
            break;
        case 'd':
        case LONGOPT_BASE+4:
#ifdef _ARCH_x86
            if (errmemd->type == ERRMEM_X86) {
                syslog(LOG_CRIT, "%s", "check command line parameters - "
                       "daemon option specified twice");
                err = -1;
                break;
            }
#else
            if (errmemd->type == ERRMEM_DAEMON) {
                syslog(LOG_CRIT, "%s", "check command line parameters - "
                       "daemon option specified twice");
                err = -1;
                break;      
            }
#endif
#ifdef _CHECK_PROCESS_
            pid_t pid = check_process(argv[0]);
            if (pid > 0) {
                syslog(LOG_CRIT, "%s %d", "a daemon is already running with pid: ", pid);
                err = -1;
                break;      
            }
#endif
            errmemd->type = ERRMEM_DAEMON;
            if (optarg)
                errmemd->frontend = strdup(optarg);
            break;
        case 'u':
        case LONGOPT_BASE+5:
            if (~w_high) {
                syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                       "watermark high defined twice: 1st: ", w_high, "2nd: ", atoi(optarg));
                err = -1;
                break;      
            }
            errno = 0;
            w_high = strtol(optarg, NULL, 0);
            if (errno) {
                w_high = ~0;
                syslog(LOG_CRIT, "%s %s %s %d", "check command line parameters - "
                       "watermark high has invalid value = ", optarg, " - error = ", errno);
                err = -1;
                break;
            }       
            break;
        case 'l':
        case LONGOPT_BASE+6:
            if (~w_low) {
                syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                       "watermark low defined twice: 1st: ", w_low, "2nd: ", atoi(optarg));
                err = -1;
                break;      
            }
            errno = 0;
            w_low = strtol(optarg, NULL, 0);
            if (errno) {
                w_low = ~0;
                syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                       "watermark low has invalid value: ", optarg);
                err = -1;
                break;
            }       
            break;
        case 'a':
        case LONGOPT_BASE+7:
            if (errmemd->tcpip_address) {
                syslog(LOG_CRIT, "%s %s %s %s", "check command line parameters - "
                       "tcp address defined twice: 1st: ", errmemd->tcpip_address, "2nd: ", optarg);
                err = -1;
                break;
            }
            errmemd->tcpip_address = check_tcpip_address(optarg);
            if (!errmemd->tcpip_address)
                err = -1;
            break;
        case 'p':
        case LONGOPT_BASE+8:
            if (errmemd->port != 0) {
                syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                       "tcp port defined twice: 1st: ", errmemd->port, " 2nd: ", atoi(optarg));
                err = -1;
                break;      
            }
            errno = 0;
            errmemd->port = strtol(optarg, NULL, 0);
            if (errno) {
                errmemd->port = ~0;
                syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                       "tcp port has invalid value: ", optarg);
                err = -1;
                break;
            }
            break;
        case 'k':
        case LONGOPT_BASE+9:
            if (errmemd->out_ack) {
                syslog(LOG_CRIT, "%s", "check command line parameters - option "
                       "-k (--keep-msg-in-driver) defined twice");
                err = -1;
                break;      
            }
            errmemd->out_ack = 1;
            break;
        case 's':
        case 'r':
        case LONGOPT_BASE+10:
        case LONGOPT_BASE+28:
            if ((opt == 's') || (opt == LONGOPT_BASE+10)) {
                eb.type = BLOCK_RAW;
                if (!optarg) {
                    syslog(LOG_CRIT, "%s", "option -s (--block-storage) has no "
                           "argument defined");
                    err = -1;
                    break;
                }
            }
            if ((opt == 'r') || (opt == LONGOPT_BASE+28)) {
                eb.type = BLOCK_FLASH_RAW; /* only to indicate a flash device */
                if (!optarg) {
                    syslog(LOG_CRIT, "%s", "option -r (--flash-storage) has no "
                           "argument defined");
                    err = -1;
                    break;
                }
            }
            eb.w_width        = ~0;
            eb.persistent     = ~0;
            eb.access_options =  OPT_ACC_NONE;
            eb.name = strdup(optarg);
            while (!repeat) {
                opterr = 0;
                opt = getopt_long(argc, argv, sh_opt, long_opt, &opt_idx);
                if (opt == -1) {
                    /* handle the end in the outer loop */
                    repeat = 1;
                    break;
                }
                /* check option argument because getopt does not do it */
                if (opt != ':' && opt != '?') {
                    s_opt = opt;
                    opt = check_option_arg(opt, optarg, sh_opt_arg,
                                           long_opt);
                    if (opt == ':') {
                        optopt = s_opt;
                        err = -1;
                    }
                }
                switch(opt) {
                case 'o':
                case LONGOPT_BASE+11:
                    if (eb.type == BLOCK || eb.type == BLOCK_FLASH) {
                            syslog(LOG_CRIT, "%s", "check command line parameters - "
                                   "option -o (--access_options) defined twice");
                            err = -1;
                            break;
                    }
                    if (eb.wait != 0) {
                            syslog(LOG_CRIT, "%s", "check command line parameters - "
                                   "option -E (--erase-wait-time) not supported with option -o (--access_options)");
                            err = -1;
                            break;
                    }
                    if (eb.retry != 0) {
                            syslog(LOG_CRIT, "%s", "check command line parameters - "
                                   "option -E (--erase-wait-time) not supported with option -o (--access_options)");
                            err = -1;
                            break;
                    }
                    if (eb.type == BLOCK_RAW)
                        eb.type = BLOCK;
                    if (eb.type == BLOCK_FLASH_RAW)
                        eb.type = BLOCK_FLASH;
                    if (optarg) { 
                        if (eb.access_options) { /* This rather should be handled above - but anyway */
                            syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                                   "option -o (--access_options) defined twice: 1st = ", eb.access_options,
                                   "2nd = ", atoi(optarg));
                            err = -1;
                            break;                          
                        }
                        errno = 0;
                        eb.access_options = strtol(optarg, NULL, 0);
                        if (errno) {
                            eb.access_options = OPT_ACC_NONE;
                            syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                                   "access_type is not supported: ", optarg);
                            err = -1;
                            break;
                        }
                        if (eb.access_options >= OPT_ACC_END) {
                            syslog(LOG_CRIT, "%s %s", "check command line parameters - "
                                   "access_type is invalid: ", optarg);
                            err = -1;
                            break;
                        }
                        if (eb.access_options && eb.type == BLOCK_FLASH) {
                            syslog(LOG_CRIT, "%s %s", "check command line parameters - "
                                   "access_options (-o [0..3]) not supported for flash devices. Given option is: ", optarg);
                            err = -1;
                            break;
                        }
                    } else
                        eb.access_options = OPT_ACC_NONE;
                    break;
                case 'b':
                case LONGOPT_BASE+12:
                    if (eb.size != 0) {
                        syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                               "option -b (--bs) defined twice: 1st = ", eb.size,  
                               "2nd = ", atoi(optarg));
                        err = -1;
                        break;                          
                    }
                    errno = 0;
                    eb.size = strtol(optarg, NULL, 0);
                    if (errno) {
                        eb.size = ~0;
                        syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                               "blocksize has invalid value: ", optarg);
                        err = -1;
                        break;
                    }
                    break;
                case 'c':
                case LONGOPT_BASE+13:
                    if (eb.cnt != 0) {
                        syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                               "option -c (--cnt) defined twice: 1st = ", eb.cnt,
                               " 2nd = ", atoi(optarg));
                        err = -1;
                        break;      
                    }
                    errno = 0;
                    eb.cnt = strtol(optarg, NULL, 0);
                    if (errno) {
                        eb.cnt = ~0;
                        syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                               "invalid value for number of blocks", optarg);
                        err = -1;
                        break;
                    }
                    break;
                case 'e':
                case LONGOPT_BASE+14:
                    if (~eb.persistent) {
                        syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                               "option -e (--persistent) defined twice: 1st = ", eb.persistent,
                               " 2nd = ", atoi(optarg));
                        err = -1;
                        break;      
                    }
                    errno = 0;
                    eb.persistent = strtol(optarg, NULL, 0);
                    if (errno) {
                        eb.persistent = ~0;
                        syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                                "invalid value for number of non-overwritable blocks", optarg);
                        err = -1;
                        break;
                    }
                    break;
                case 'w':
                case LONGOPT_BASE+15:
                    if (~eb.w_width) {
                        syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                               "option -w (--write-size) defined twice: 1st = ", eb.w_width, " 2nd = ",
                                atoi(optarg));
                        err = -1;
                        break;      
                    }
                    errno = 0;
                    eb.w_width = strtol(optarg, NULL, 0);
                    if (errno) {
                        eb.w_width = ~0;
                        fprintf(stderr, "%s %s %m", "check command line parameters - "
                                "option -w (--write-size) has invalid value: ", optarg);
                        err = -1;
                        break;
                    }
                    break;
                case 'E':
                case LONGOPT_BASE+29:
                    if (eb.type != BLOCK_FLASH_RAW) {
                        syslog(LOG_CRIT, "%s", "check command line parameters - "
                               "option -E (--erase-wait-time) not supported for none bare flash devices");
                        err = -1;
                        break;                          
                    }
                    if (eb.wait) {
                        syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                               "option -E (--erase-wait-time) defined twice: 1st = ", eb.wait, " 2nd = ",
                                atoi(optarg));
                        err = -1;
                        break;      
                    }
                    errno = 0;
                    eb.wait = strtol(optarg, NULL, 0);
                    if (errno) {
                        eb.wait = 0;
                        fprintf(stderr, "%s %s %m", "check command line parameters - "
                                "option -E (--erase-wait-time) has invalid value: ", optarg);
                        err = -1;
                        break;
                    }
                    break;
                case 'R':
                case LONGOPT_BASE+30:
                    if (eb.type != BLOCK_FLASH_RAW) {
                        syslog(LOG_CRIT, "%s", "check command line parameters - "
                               "option -R (--erase-retry) not supported for none bare flash devices");
                        err = -1;
                        break;                          
                    }
                    if (eb.retry) {
                        syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                               "option -R (--erase-retry) defined twice: 1st = ", eb.retry, " 2nd = ",
                                atoi(optarg));
                        err = -1;
                        break;      
                    }
                    errno = 0;
                    eb.retry = strtol(optarg, NULL, 0);
                    if (errno) {
                        eb.retry = 0;
                        fprintf(stderr, "%s %s %m", "check command line parameters - "
                                "option -R (--erase-retry) has invalid value: ", optarg);
                        err = -1;
                        break;
                    }
                    break;
                case 'C':
                case LONGOPT_BASE+27:
                    if (eb.command) {
                        syslog(LOG_CRIT, "%s %s %s %s", "check command line parameters - "
                               "option -C (--cmd) is defined twice: 1st = ", eb.command, " 2nd = ", optarg);
                        err = -1;
                        break;
                    }
                    if (!optarg) {
                        syslog(LOG_CRIT, "%s", "option -C (--cmd) has no argument defined");
                        err = -1;
                        break;
                    }
                    eb.command = strdup(optarg);
                    if (!eb.command) {
                        syslog(LOG_CRIT, "%s %s", "cannot get command from commandline", optarg);
                        err = -1;
                        break;                  
                    }
                    break;
                default:
                    /* this option will be handled in the outer loop */
                    repeat = 1;
                    break;
                }
                /* Leave inner while loop in case of error or if the 
                 * first option is found which has not to be handled here
                 */
                if(err || repeat)
                    break;
            } /* end of inner while loop to determine options for block devices */
            /* parameters will be checked at the end of command line parsing 
             * because not till then we definitly know the operation mode */
            if (!err) {
                eb.nr = ++nr;
                err = create_backend_structure(errmemd, &eb);
            }
            if (eb.name)
                free(eb.name);
            if (eb.command)
                free(eb.command);
            break;
        case 'f':
        case LONGOPT_BASE+16:
            eb.type = FILE_FS;
            eb.persistent = ~0;
            if (!optarg) {
                syslog(LOG_CRIT, "%s",  "option -f (--file-storage) has no argument defined");
                err = -1;
                break;
            }
            eb.name = strdup(optarg);
            while (!repeat) {
                opterr = 0;
                opt = getopt_long(argc, argv, sh_opt, long_opt, &opt_idx);
                if (opt == -1) {
                    /* handle the end in the outer loop */
                    repeat = 1;
                    break;
                }

                /* check option argument because getopt does not do it */
                if (opt != ':' && opt != '?') {
                    s_opt = opt;
                    opt = check_option_arg(opt, optarg, sh_opt_arg,
                                           long_opt);
                    if (opt == ':') {
                        optopt = s_opt;
                        err = -1;
                    }
                }
                switch(opt) {
                case 'b':
                case LONGOPT_BASE+17:
                    if (eb.size != 0) {
                        syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                               "option -b (--bs) defined twice: 1st = ", eb.size,  
                               "2nd = ", atoi(optarg));
                        err = -1;
                        break;      
                    }
                    errno = 0;
                    eb.size = strtol(optarg, NULL, 0);
                    if (errno) {
                        eb.size = ~0;
                        syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                               "blocksize has invalid value: ", optarg);
                        err = -1;
                        break;
                    }
                    break;
                case 'c':
                case LONGOPT_BASE+18:
                    if (eb.cnt != 0) {
                        syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                               "option -c (--cnt) defined twice: 1st = ", eb.cnt,
                               " 2nd = ", atoi(optarg));
                        err = -1;
                        break;      
                    
                    }
                    errno = 0;
                    eb.cnt = strtol(optarg, NULL, 0);
                    if (errno) {
                        eb.cnt = ~0;
                        syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                               "invalid value for number of blocks", optarg);
                        err = -1;
                        break;
                    }
                    break;
                case 'e':
                case LONGOPT_BASE+19:
                    if (~eb.persistent) {
                        syslog(LOG_CRIT, "%s %d %s %d", "check command line parameters - "
                               "option -e (--persistent) defined twice: 1st = ", eb.persistent,
                               " 2nd = ", atoi(optarg));
                        err = -1;
                        break;      
                    }
                    errno = 0;
                    eb.persistent = strtol(optarg, NULL, 0);
                    if (errno) {
                        eb.persistent = ~0;
                        syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                                "invalid value for number of non-overwritable blocks", optarg);
                        err = -1;
                        break;
                    }
                    break;
                case 'C':
                case LONGOPT_BASE+27:
                    if (eb.command) {
                        syslog(LOG_CRIT, "%s %s %s %s", "check command line parameters - "
                               "option -C (--cmd) is defined twice: 1st = ", eb.command, " 2nd = ", optarg);
                        err = -1;
                        break;
                    }
                    if (!optarg) {
                        syslog(LOG_CRIT, "%s", "option -C (--cmd) has no argument defined");
                        err = -1;
                        break;
                    }
                    eb.command = strdup(optarg);
                    if (!eb.command) {
                        syslog(LOG_CRIT, "%s %s", "cannot get command from commandline", optarg);
                        err = -1;
                        break;                  
                    }
                    break;
                default:
                    /* this option will be handled in the outer loop */
                    repeat = 1;
                    break;
                }
                /* Leave inner while loop in case of error or if the 
                 * first option is found which has not to be handled here
                 */
                if(err || repeat)
                    break;
            } /* end of inner while loop to determine options for block devices */
            if (!err) {
                eb.nr = ++nr;
                err = create_backend_structure(errmemd, &eb);
            }
            if (eb.name)
                free(eb.name);
            if (eb.command)
                free(eb.command);
            break;
        case 'g':
        case LONGOPT_BASE+21:
                eb.type = ERRMEM_DLT_OUT;
                err = create_backend_structure(errmemd, &eb);
            break;
        case 'x':
        case LONGOPT_BASE+22:
#ifdef _ARCH_x86
            if (errmemd->type == ERRMEM_X86_NO_DEAMON) {
                syslog(LOG_CRIT, "%s", "check command line parameters - "
                       "option -x (--x86_ndaemon_dbg specified twice");
                err = -1;
                break;      
            }
            errmemd->type = ERRMEM_X86_NO_DEAMON;
#else
            syslog(LOG_CRIT, "%s", "option -x (--x86_ndaemon_dbg option not supported for ARM");
            err = -1;
#endif
            break;
        case 'y':
        case LONGOPT_BASE+23:
#ifdef _ARCH_x86
            if (fill_path) {
                syslog(LOG_CRIT, "%s %s %s %s", "option -y (--x86_fill_source) defined "
                       "twice: 1st = ", fill_path, "2nd = ", optarg);
                err = -1;
                break;
            }
            if (!optarg) {
                syslog(LOG_CRIT, "%s", "option -y has no argument defined");
                err = -1;
                break;
            }
            fill_path = strdup(optarg);
#else
            syslog(LOG_CRIT, "%s", "option -y (--x86_fill_source) not supported for ARM");
            err = -1;
#endif
            break;
        case 'o':
        case LONGOPT_BASE+11:
            if (optopt < LONGOPT_BASE)
                syslog(LOG_CRIT, "%s %s %s", "short option ", (char*)(&opt), 
                       "prohibited without the use of the option -s -r");
            else
                syslog(LOG_CRIT, "%s %s %s", "long option ", long_opt[opt - LONGOPT_BASE].name,
                        "prohibited without the use of the options -s -r");
            err = -1;
            break;
        case 'b':
        case LONGOPT_BASE+12:
        case 'c':
        case LONGOPT_BASE+13:
        case 'e':
        case LONGOPT_BASE+14:
            if (optopt < LONGOPT_BASE)
                syslog(LOG_CRIT, "%s %s %s", "short option ", (char*)(&opt), "prohibited without "
                       "the use of one of the options -s -r -f");
            else
                syslog(LOG_CRIT, "%s %s %s", "long option ", long_opt[opt - LONGOPT_BASE].name,
                       "prohibited without the use of one of the options -s -r -f");
            err = -1;
            break;
        case 'C':
        case LONGOPT_BASE+27:
            if (optopt < LONGOPT_BASE)
                syslog(LOG_CRIT, "%s %s %s", "short option ",  (char*)(&opt), 
                        "prohibited without the use of one of the options -s -r -f");
            else
                syslog(LOG_CRIT, "%s %s %s", "long option ", long_opt[opt - LONGOPT_BASE].name,
                       " prohibited without the use of one of the options -s -r -f");
            err = -1;
            break;
        case 'w':
        case LONGOPT_BASE+15:
            if (optopt < LONGOPT_BASE)
                syslog(LOG_CRIT, "%s %s %s", "short option ", (char*)(&opt),
                       "prohibited without the use of option -s or -r");
            else
                syslog(LOG_CRIT, "%s %s %s", "long option ", long_opt[opt - LONGOPT_BASE].name,
                        "prohibited without the use of option -s or -r");
            err = -1;
            break;
        case 'E':
        case LONGOPT_BASE+29:
            if (optopt < LONGOPT_BASE)
                syslog(LOG_CRIT, "%s %s %s", "short option ", (char*)(&opt),
                       "prohibited without the use of option -r");
            else
                syslog(LOG_CRIT, "%s %s %s", "long option ", long_opt[opt - LONGOPT_BASE].name,
                        "prohibited without the use of option -r");
            err = -1;
            break;
        case 'R':
        case LONGOPT_BASE+30:
            if (optopt < LONGOPT_BASE)
                syslog(LOG_CRIT, "%s %s %s", "short option ", (char*)(&opt),
                       "prohibited without the use of option -r");
            else
                syslog(LOG_CRIT, "%s %s %s", "long option ", long_opt[opt - LONGOPT_BASE].name,
                        "prohibited without the use of option -r");
            err = -1;
            break;
        case 'n':
        case LONGOPT_BASE+25:
            if (nr_dump != 0) {
                syslog(LOG_CRIT, "%s %d %s %d", "check option arguments - parameter"
                       " -n (--number-rows) defined twice: 1st = ", nr_dump, "2nd = ", atoi(optarg));
                err = -1;
                break;
            }
            errno = 0;
            nr_dump = strtol(optarg, NULL, 0);
            if (errno) {
                nr_dump = 0;
                syslog(LOG_CRIT, "%s %s %m", "check command line parameters - "
                        "number of lines to dump has invalid value: ", optarg);
                err = -1;
                break;
            }
            break;
        case 'z':
        case LONGOPT_BASE+26:
            if (filter) {
                syslog(LOG_CRIT, "%s %s %s %s", "check option arguments - parameter"
                       " -z (--filtered) defined twice: 1st = ", filter, "2nd = ", optarg);
                err = -1;
                break;
            }
            if (!optarg) {
                syslog(LOG_CRIT, "%s", "option -z (--filtered) has no argument defined");
                err = -1;
                break;
            }
            filter = strdup(optarg);
            if (!filter) {
                syslog(LOG_CRIT, "%s %s", "cannot get mem to store filter pattern: ", optarg);
                err = -1;
                break;
            }                       
            break;
        case 'O':
        case LONGOPT_BASE+31:
            if (errmemd->out_once) {
                syslog(LOG_CRIT, "%s", "check command line parameters - option "
                       "-O (--once) defined twice");
                err = -1;
                break;      
            }
            errmemd->out_once = 1;
            break;
        case ':':
            if (optopt < LONGOPT_BASE)
                syslog(LOG_CRIT, "%s %s %s", "short option ", (char*)(&optopt), "requires parameter");
            else
                syslog(LOG_CRIT, "%s %s %s", "long option" , long_opt[optopt - LONGOPT_BASE].name,
                       "requires parameter");
            err = -1;
            break;
        case '?':
            syslog(LOG_CRIT, "%s %s", "option not defined: ",(char*)(&optopt));
            err = -1;
            break;
        default:
            syslog(LOG_CRIT, "%s %d", "getopt returned unknown character code ", opt);
            err = -1;
            break;
        } /* end switch */
        if (err)
            break;
    } /* end while */

    /*final check of collected command line parameters */
    if (verbose) {
        int32_t i = 1;
        syslog(LOG_INFO, "%s", "got following options/values from command line");
        for (; i < argc; i++){
            syslog(LOG_INFO, "%s %s", "option/value : ", argv[i]);
        }
    }
    
    if (errmemd) {
        // handle some signals
        install_signal_handler();

        if (err >= 0)
            create_backends(errmemd);

        if (err >= 0) {
            if (info) {
                errmem_backend_info(errmemd->backends, 2);
                errmem_backend_info(errmemd->outputs, 2);
                errmem_smoketest(errmemd->frontend);
            }
        }

        if (err >= 0) {
            if (errmemd->type != ERRMEM_CLIENT && errmemd->type != ERRMEM_UNKNOWN) {
                /* create frontend interface for device named errmemd->frontend */
                errmemd->ifc  = (ErrmemInterface_t*)errmem_create(errmemd->frontend);
                if (!errmemd->ifc) {
                    if (errmemd->frontend)
                        syslog(LOG_CRIT, "%s %s",
                               "can't create interface for device",
                               errmemd->frontend);
                    else
                        syslog(LOG_CRIT, "%s",
                               "can't create interface for device /dev/errmem");
                    err = -1;
                }
            
                if (err >= 0)
                    err = errmem_configure_frontend(errmemd, w_low, w_high);

                if (err > 0) {
                    if (info) {
                        syslog(LOG_INFO, "%s", "Frontend-Settings changed to: \n");
                        errmem_info(errmemd->ifc, stdout);
                    }
                }

                if (err >= 0) {
                    errmemd->is = calloc(1, sizeof(ErrmemSocket_t));
                    if (!errmemd->is) {
                        syslog(LOG_CRIT, "%s", "failed to allocate memory for server socket");
                        err = -1;
                    }
                }
                if (err >= 0)
                    /* create socket server for clients */
                    err = errmem_create_server(errmemd->is);
                if (err >= 0) {
                    /* prepare communication to frontend 
                     * if backends specified and available */
                    if (errmemd->backends || errmemd->outputs)
                        err = errmem_connect_frontend(errmemd);
                    else
                        syslog(LOG_INFO, "%s", "Daemon not connected to "
                               "frontend due to missing backends");

                    if (!errmemd->backends)
                        syslog(LOG_INFO, "%s", "!! No persistent backends "
                               "available !!. Messages from frontend will "
                               "not be made persistent.");
                    if (!err) {
						errmemd->flags |= ERRMEMD_START;
                        process(errmemd);
					}
                }
            } else {
                ErrmemBackend_t* cur  = errmemd->backends;
                char* cmd = NULL;
                while(cur){
                    if (cur->get_command)
                        cmd = cur->get_command(cur);
                    if (cmd){
                        if (!strncmp((const char*)cmd, (const char*)"dump", strlen("dump"))) {
                            ErrmemBackendDev_t eb = {.type = ERRMEM_STD_OUT, .debug = 0};
                            ErrmemBackend_t* out  = errmem_backend_factory(&eb);
                            if (out && cur->dump) {
                                cur->dump(cur, out, NULL, filter, 0, nr_dump);
                                if (out->destroy)
                                    out->destroy(out);
                            }
                        }
                        else if (!strncmp((const char*)cmd, (const char*)"ddump", strlen("ddump"))) {
                            ErrmemBackendDev_t eb = {.type = ERRMEM_STD_OUT, .debug = 1};
                            ErrmemBackend_t* out  = errmem_backend_factory(&eb);
                            if (out && cur->dump) {
                                cur->dump(cur, out, NULL, filter, 0, nr_dump);
                                if (out->destroy)
                                    out->destroy(out);
                            }
                        }
                        else if (!strncmp((const char*)cmd, (const char*)"erase", strlen("erase"))) {
                            int rc;
                            rc = errmem_lib_erase(cur);
                            if(!rc) {
                                cur = cur->next;
                                continue;
                            }
                            /* errmem deamon is not running
                             * erase through dirty method */
                            if (cur->erase)
                                cur->erase(cur);
                            syslog(LOG_CRIT,"%s",
                                           "errmemd erased backend in dirty way "
                                           "Please restart errmem daemon if running");
                        }
                        else
                            syslog(LOG_INFO, "%s %s", "command not supported ", cmd);
                    } 
                    cur = cur->next;
                }
            }
        }
        errmemd_destroy(errmemd);
    }

    /* free local resources */

    if (filter)
        free(filter);

    syslog(LOG_INFO, "%s", "Error memory backend exits");

    closelog();
    return 0;
}
